@extends('layouts.app')
@section('title')
    Criar Usuário
@endsection
@section('content')
    <x-page-title title="Usuários" subtitle="Criar Usuário" />

    <div class="card mt-4">
        <div class="card-body">
            <div id="alert" class="d-none mb-4 p-4 rounded-lg text-white"></div>
            <form id="create-user-form">
                @csrf
                <div class="mb-4">
                    <label for="name" class="form-label">Nome</label>
                    <input type="text" name="name" class="form-control" required>
                </div>
                <div class="mb-4">
                    <label for="email" class="form-label">Email</label>
                    <input type="email" name="email" class="form-control" required>
                </div>
                <div class="mb-4">
                    <label for="password" class="form-label">Senha</label>
                    <input type="password" name="password" class="form-control" required>
                </div>
                <div class="mb-4">
                    <label for="password_confirmation" class="form-label">Confirmar Senha</label>
                    <input type="password" name="password_confirmation" class="form-control" required>
                </div>
                <div class="mb-4">
                    <label for="role" class="form-label">Tipo de Usuário</label>
                    <select name="role" class="form-select" required>
                        <option value="admin" selected>Admin</option>
                        <option value="architect">Arquiteto</option>
                        <option value="store_manager">Gerente de Loja</option>
                    </select>
                </div>

                <!-- Campos específicos para arquitetos -->
                <div id="architect-fields">
                    <div class="mb-4">
                        <label for="cpf" class="form-label">CPF</label>
                        <input type="text" name="cpf" class="form-control" id="cpf">
                    </div>
                    <div class="mb-4">
                        <label for="cep" class="form-label">CEP</label>
                        <input type="text" name="cep" class="form-control" id="cep">
                    </div>
                    <div class="mb-4">
                        <label for="rua" class="form-label">Rua</label>
                        <input type="text" name="rua" class="form-control" id="rua">
                    </div>
                    <div class="mb-4">
                        <label for="numero" class="form-label">Número</label>
                        <input type="text" name="numero" class="form-control" id="numero">
                    </div>
                    <div class="mb-4">
                        <label for="bairro" class="form-label">Bairro</label>
                        <input type="text" name="bairro" class="form-control" id="bairro">
                    </div>
                    <div class="mb-4">
                        <label for="cidade" class="form-label">Cidade</label>
                        <input type="text" name="cidade" class="form-control" id="cidade">
                    </div>
                    <div class="mb-4">
                        <label for="estado" class="form-label">Estado</label>
                        <input type="text" name="estado" class="form-control" id="estado">
                    </div>
                    <div class="mb-4">
                        <label for="phone" class="form-label">Telefone</label>
                        <input type="text" name="phone" class="form-control" id="phone">
                    </div>
                    <div class="mb-4">
                        <label for="birth_date" class="form-label">Data de Nascimento</label>
                        <input type="date" name="birth_date" class="form-control">
                    </div>
                </div>

                <!-- Campos específicos para gerentes de loja -->
                <div id="store-manager-fields" style="display: none;">
                    <div class="mb-4">
                        <label for="store_id" class="form-label">Loja</label>
                        <select name="store_id" class="form-select" id="store_id">
                            @foreach($stores as $store) <!-- Assumindo que $stores é passado para a view -->
                                <option value="{{ $store->id }}">{{ $store->name }}</option>
                            @endforeach
                        </select>
                    </div>
                    <div class="mb-4">
                        <label for="phone" class="form-label">Telefone</label>
                        <input type="text" name="phone" class="form-control" id="store-manager-phone">
                    </div>
                </div>

                <div class="d-flex justify-content-between">
                    <button type="submit" id="submit-button" class="btn btn-primary">Criar</button>
                    <div id="loader" class="spinner-border text-primary d-none" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function () {
            const roleSelect = document.querySelector('select[name="role"]');
            const architectFields = document.getElementById('architect-fields');
            const storeManagerFields = document.getElementById('store-manager-fields');
            const cpfInput = document.getElementById('cpf');
            const cepInput = document.getElementById('cep');
            const phoneInput = document.getElementById('phone');
            const storeManagerPhoneInput = document.getElementById('store-manager-phone');

            function toggleFields() {
                if (roleSelect.value === 'architect') {
                    architectFields.style.display = 'block';
                    storeManagerFields.style.display = 'none';
                } else if (roleSelect.value === 'store_manager') {
                    architectFields.style.display = 'none';
                    storeManagerFields.style.display = 'block';
                } else {
                    architectFields.style.display = 'none';
                    storeManagerFields.style.display = 'none';
                }
            }

            roleSelect.addEventListener('change', toggleFields);
            toggleFields(); // Inicializar com a configuração correta

            // Máscaras de input
            cpfInput.addEventListener('input', function () {
                cpfInput.value = cpfInput.value.replace(/\D/g, '')
                    .replace(/(\d{3})(\d)/, '$1.$2')
                    .replace(/(\d{3})(\d)/, '$1.$2')
                    .replace(/(\d{3})(\d{1,2})$/, '$1-$2');
            });

            cepInput.addEventListener('input', function () {
                cepInput.value = cepInput.value.replace(/\D/g, '').replace(/(\d{5})(\d)/, '$1-$2');
            });

            phoneInput.addEventListener('input', function () {
                phoneInput.value = phoneInput.value.replace(/\D/g, '')
                    .replace(/(\d{2})(\d)/, '($1) $2')
                    .replace(/(\d{5})(\d{1,4})$/, '$1-$2');
            });

            storeManagerPhoneInput.addEventListener('input', function () {
                storeManagerPhoneInput.value = storeManagerPhoneInput.value.replace(/\D/g, '')
                    .replace(/(\d{2})(\d)/, '($1) $2')
                    .replace(/(\d{5})(\d{1,4})$/, '$1-$2');
            });

            cepInput.addEventListener('blur', function () {
                const cep = cepInput.value.replace(/\D/g, '');
                if (cep !== '') {
                    const validacep = /^[0-9]{8}$/;
                    if (validacep.test(cep)) {
                        fetch(`https://viacep.com.br/ws/${cep}/json/`)
                            .then(response => response.json())
                            .then(data => {
                                if (!data.erro) {
                                    document.getElementById('rua').value = data.logradouro;
                                    document.getElementById('bairro').value = data.bairro;
                                    document.getElementById('cidade').value = data.localidade;
                                    document.getElementById('estado').value = data.uf;
                                } else {
                                    alert("CEP não encontrado.");
                                }
                            })
                            .catch(error => console.error('Erro ao buscar CEP:', error));
                    } else {
                        alert("Formato de CEP inválido.");
                    }
                }
            });

            function validarCPF(cpf) {
                cpf = cpf.replace(/[^\d]+/g,'');
                if (cpf === '') return false;
                if (cpf.length !== 11 ||
                    cpf === "00000000000" || cpf === "11111111111" || cpf === "22222222222" ||
                    cpf === "33333333333" || cpf === "44444444444" || cpf === "55555555555" ||
                    cpf === "66666666666" || cpf === "77777777777" || cpf === "88888888888" ||
                    cpf === "99999999999")
                    return false;
                let add = 0;
                for (let i = 0; i < 9; i++) add += parseInt(cpf.charAt(i)) * (10 - i);
                let rev = 11 - (add % 11);
                if (rev === 10 || rev === 11) rev = 0;
                if (rev !== parseInt(cpf.charAt(9))) return false;
                add = 0;
                for (let i = 0; i < 10; i++) add += parseInt(cpf.charAt(i)) * (11 - i);
                rev = 11 - (add % 11);
                if (rev === 10 || rev === 11) rev = 0;
                return rev === parseInt(cpf.charAt(10));
            }

            document.getElementById('create-user-form').addEventListener('submit', function (event) {
                event.preventDefault();

                if (roleSelect.value === 'architect') {
                    const cpf = cpfInput.value;
                    if (cpf && !validarCPF(cpf)) {
                        alert('CPF inválido');
                        return;
                    }
                }

                const form = this;
                const formData = new FormData(form);

                document.getElementById('submit-button').classList.add('d-none');
                document.getElementById('loader').classList.remove('d-none');

                const alert = document.getElementById('alert');
                alert.classList.add('d-none');
                alert.classList.remove('bg-success', 'bg-danger');
                alert.innerHTML = '';

                fetch('{{ route('admin.users.store') }}', {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Accept': 'application/json',
                    },
                    body: formData
                }).then(response => {
                    document.getElementById('submit-button').classList.remove('d-none');
                    document.getElementById('loader').classList.add('d-none');

                    if (!response.ok) {
                        return response.json().then(data => {
                            throw new Error(JSON.stringify(data.errors));
                        });
                    }
                    return response.json().then(data => ({
                        status: response.status,
                        body: data
                    }));
                }).then(response => {
                    if (response.status === 201) {
                        window.location.href = '{{ route('admin.users.index') }}?message=success';
                    } else if (response.status === 422) {
                        alert.classList.add('bg-danger');
                        alert.innerHTML = Object.values(response.body.errors).map(error => error.join('<br>')).join('<br>');
                        window.scrollTo({ top: 0, behavior: 'smooth' });
                    } else {
                        alert.classList.add('bg-danger');
                        alert.innerHTML = response.body.message || 'Ocorreu um erro ao criar o usuário.';
                        window.scrollTo({ top: 0, behavior: 'smooth' });
                    }
                    alert.classList.remove('d-none');
                }).catch(error => {
                    document.getElementById('submit-button').classList.remove('d-none');
                    document.getElementById('loader').classList.add('d-none');

                    let errorMessage;
                    try {
                        errorMessage = JSON.parse(error.message);
                        errorMessage = Object.values(errorMessage).flat().join('<br>');
                    } catch {
                        errorMessage = error.message;
                    }
                    alert.classList.add('bg-danger');
                    alert.innerHTML = 'Ocorreu um erro ao criar o usuário: ' + errorMessage;
                    alert.classList.remove('d-none');
                    window.scrollTo({ top: 0, behavior: 'smooth' });
                });
            });
        });
    </script>
@endsection

@push('script')
    <!--plugins-->
    <script src="{{ URL::asset('build/plugins/perfect-scrollbar/js/perfect-scrollbar.js') }}"></script>
    <script src="{{ URL::asset('build/plugins/metismenu/metisMenu.min.js') }}"></script>
    <script src="{{ URL::asset('build/plugins/simplebar/js/simplebar.min.js') }}"></script>
    <script src="{{ URL::asset('build/js/main.js') }}"></script>
@endpush
