@extends('layouts.app')
@section('title')
    Editar Arquiteto
@endsection
@section('content')
    <x-page-title title="Arquitetos" subtitle="Editar Arquiteto" />

    <div class="card mt-4">
        <div class="card-body">
            <div id="alert" class="d-none mb-4 p-4 rounded-lg text-white"></div>
            <form id="edit-architect-form" action="{{ route('admin.architects.update', $architect->id) }}" method="POST">
                @csrf
                @method('PUT')
                <div class="mb-4">
                    <label for="name" class="form-label">Nome</label>
                    <input type="text" name="name" class="form-control" value="{{ $architect->name }}" required>
                </div>
                <div class="mb-4">
                    <label for="email" class="form-label">Email</label>
                    <input type="email" name="email" class="form-control" value="{{ $architect->email }}" required>
                </div>
                <div class="mb-4">
                    <label for="cpf" class="form-label">CPF</label>
                    <input type="text" name="cpf" class="form-control" id="cpf" value="{{ $architect->architectProfile->cpf }}">
                    <div id="cpf-invalid-feedback" class="invalid-feedback"></div>
                </div>
                <div class="mb-4">
                    <label for="cep" class="form-label">CEP</label>
                    <input type="text" name="cep" class="form-control" id="cep" value="{{ $architect->architectProfile->cep }}">
                </div>
                <div class="mb-4">
                    <label for="rua" class="form-label">Rua</label>
                    <input type="text" name="rua" class="form-control" id="rua" value="{{ $architect->architectProfile->rua }}">
                </div>
                <div class="mb-4">
                    <label for="numero" class="form-label">Número</label>
                    <input type="text" name="numero" class="form-control" id="numero" value="{{ $architect->architectProfile->numero }}">
                </div>
                <div class="mb-4">
                    <label for="bairro" class="form-label">Bairro</label>
                    <input type="text" name="bairro" class="form-control" id="bairro" value="{{ $architect->architectProfile->bairro }}">
                </div>
                <div class="mb-4">
                    <label for="cidade" class="form-label">Cidade</label>
                    <input type="text" name="cidade" class="form-control" id="cidade" value="{{ $architect->architectProfile->cidade }}">
                </div>
                <div class="mb-4">
                    <label for="estado" class="form-label">Estado</label>
                    <input type="text" name="estado" class="form-control" id="estado" value="{{ $architect->architectProfile->estado }}">
                </div>
                <div class="mb-4">
                    <label for="phone" class="form-label">Telefone</label>
                    <input type="text" name="phone" class="form-control" id="phone" value="{{ $architect->architectProfile->phone }}">
                </div>
                <div class="mb-4">
                    <label for="birth_date" class="form-label">Data de Nascimento</label>
                    <input type="date" name="birth_date" class="form-control" value="{{ $architect->architectProfile->birth_date }}">
                </div>

                <!-- Checkbox para alterar senha -->
                <div class="mb-4 form-check">
                    <input type="checkbox" class="form-check-input" id="change-password-checkbox">
                    <label class="form-check-label" for="change-password-checkbox">Alterar Senha</label>
                </div>

                <!-- Campos de senha ocultos por padrão -->
                <div id="password-fields" style="display: none;">
                    <div class="mb-4">
                        <label for="password" class="form-label">Senha</label>
                        <input type="password" name="password" class="form-control">
                        <small class="form-text text-muted">Deixe em branco para manter a senha atual.</small>
                    </div>
                    <div class="mb-4">
                        <label for="password_confirmation" class="form-label">Confirmar Senha</label>
                        <input type="password" name="password_confirmation" class="form-control">
                    </div>
                </div>

                <div class="d-flex justify-content-between">
                    <button type="submit" id="submit-button" class="btn btn-primary">Salvar</button>
                    <div id="loader" class="spinner-border text-primary d-none" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function () {
            const changePasswordCheckbox = document.getElementById('change-password-checkbox');
            const passwordFields = document.getElementById('password-fields');
            const cpfInput = document.getElementById('cpf');
            const cepInput = document.getElementById('cep');
            const phoneInput = document.getElementById('phone');
            const cpfInvalidFeedback = document.getElementById('cpf-invalid-feedback');

            changePasswordCheckbox.addEventListener('change', function () {
                passwordFields.style.display = this.checked ? 'block' : 'none';
            });

            cpfInput.addEventListener('input', function () {
                cpfInput.value = cpfInput.value.replace(/\D/g, '')
                    .replace(/(\d{3})(\d)/, '$1.$2')
                    .replace(/(\d{3})(\d)/, '$1.$2')
                    .replace(/(\d{3})(\d{1,2})$/, '$1-$2');
            });

            cepInput.addEventListener('input', function () {
                cepInput.value = cepInput.value.replace(/\D/g, '').replace(/(\d{5})(\d)/, '$1-$2');
            });

            phoneInput.addEventListener('input', function () {
                phoneInput.value = phoneInput.value.replace(/\D/g, '')
                    .replace(/(\d{2})(\d)/, '($1) $2')
                    .replace(/(\d{5})(\d{1,4})$/, '$1-$2');
            });

            cepInput.addEventListener('blur', function () {
                const cep = cepInput.value.replace(/\D/g, '');
                if (cep !== '') {
                    const validacep = /^[0-9]{8}$/;
                    if (validacep.test(cep)) {
                        fetch(`https://viacep.com.br/ws/${cep}/json/`)
                            .then(response => response.json())
                            .then(data => {
                                if (!data.erro) {
                                    document.getElementById('rua').value = data.logradouro;
                                    document.getElementById('bairro').value = data.bairro;
                                    document.getElementById('cidade').value = data.localidade;
                                    document.getElementById('estado').value = data.uf;
                                } else {
                                    alert("CEP não encontrado.");
                                }
                            })
                            .catch(error => console.error('Erro ao buscar CEP:', error));
                    } else {
                        alert("Formato de CEP inválido.");
                    }
                }
            });

            function validarCPF(cpf) {
                cpf = cpf.replace(/[^\d]+/g,'');
                if (cpf === '') return false;
                if (cpf.length !== 11 ||
                    cpf === "00000000000" || cpf === "11111111111" || cpf === "22222222222" ||
                    cpf === "33333333333" || cpf === "44444444444" || cpf === "55555555555" ||
                    cpf === "66666666666" || cpf === "77777777777" || cpf === "88888888888" ||
                    cpf === "99999999999")
                    return false;
                let add = 0;
                for (let i = 0; i < 9; i++) add += parseInt(cpf.charAt(i)) * (10 - i);
                let rev = 11 - (add % 11);
                if (rev === 10 || rev === 11) rev = 0;
                if (rev !== parseInt(cpf.charAt(9))) return false;
                add = 0;
                for (let i = 0; i < 10; i++) add += parseInt(cpf.charAt(i)) * (11 - i);
                rev = 11 - (add % 11);
                if (rev === 10 || rev === 11) rev = 0;
                return rev === parseInt(cpf.charAt(10));
            }

            document.getElementById('edit-architect-form').addEventListener('submit', function (event) {
                event.preventDefault();

                const cpf = cpfInput.value;
                if (cpf && !validarCPF(cpf)) {
                    cpfInvalidFeedback.innerHTML = 'CPF inválido';
                    cpfInput.classList.add('is-invalid');
                    cpfInput.focus();
                    return;
                } else {
                    cpfInvalidFeedback.innerHTML = '';
                    cpfInput.classList.remove('is-invalid');
                }

                const form = this;
                const formData = new FormData(form);

                document.getElementById('submit-button').classList.add('d-none');
                document.getElementById('loader').classList.remove('d-none');

                const alert = document.getElementById('alert');
                alert.classList.add('d-none');
                alert.classList.remove('bg-success', 'bg-danger');
                alert.innerHTML = '';

                fetch(form.action, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Accept': 'application/json',
                    },
                    body: formData
                }).then(response => {
                    document.getElementById('submit-button').classList.remove('d-none');
                    document.getElementById('loader').classList.add('d-none');

                    if (!response.ok) {
                        return response.json().then(data => {
                            throw new Error(JSON.stringify(data.errors));
                        });
                    }
                    return response.json().then(data => ({
                        status: response.status,
                        body: data
                    }));
                }).then(response => {
                    if (response.status === 200) {
                        window.location.href = '{{ route('admin.architects.show', $architect->id) }}';
                    } else if (response.status === 422) {
                        alert.classList.add('bg-danger');
                        alert.innerHTML = Object.values(response.body.errors).map(error => error.join('<br>')).join('<br>');
                        window.scrollTo({ top: 0, behavior: 'smooth' });
                    } else {
                        alert.classList.add('bg-danger');
                        alert.innerHTML = response.body.message || 'Ocorreu um erro ao atualizar o arquiteto.';
                        window.scrollTo({ top: 0, behavior: 'smooth' });
                    }
                    alert.classList.remove('d-none');
                }).catch(error => {
                    document.getElementById('submit-button').classList.remove('d-none');
                    document.getElementById('loader').classList.add('d-none');

                    let errorMessage;
                    try {
                        errorMessage = JSON.parse(error.message);
                        errorMessage = Object.values(errorMessage).flat().join('<br>');
                    } catch {
                        errorMessage = error.message;
                    }
                    alert.classList.add('bg-danger');
                    alert.innerHTML = 'Ocorreu um erro ao atualizar o arquiteto: ' + errorMessage;
                    alert.classList.remove('d-none');
                    window.scrollTo({ top: 0, behavior: 'smooth' });
                });
            });
        });
    </script>
@endsection

@push('script')
    <!--plugins-->
    <script src="{{ URL::asset('build/plugins/perfect-scrollbar/js/perfect-scrollbar.js') }}"></script>
    <script src="{{ URL::asset('build/plugins/metismenu/metisMenu.min.js') }}"></script>
    <script src="{{ URL::asset('build/plugins/simplebar/js/simplebar.min.js') }}"></script>
    <script src="{{ URL::asset('build/js/main.js') }}"></script>
@endpush
