<?php

namespace App\Http\Controllers\Manager;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Point;
use App\Models\ScoringRule;
use App\Models\Store;
use App\Models\User;

class PointsController extends Controller
{
    public function index(Request $request)
    {
        // Obter a loja do gerente logado
        $storeId = auth()->user()->storeManagerProfile->store_id;

        // Iniciar a query com a condição de filtro pela loja do gerente
        $query = Point::where('store_id', $storeId);

        // Aplicar outros filtros, se fornecidos
        if ($request->architect_id) {
            $query->where('architect_id', $request->architect_id);
        }

        if ($request->admin_id) {
            $query->where('admin_id', $request->admin_id);
        }

        if ($request->start_date) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }

        if ($request->end_date) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }

        if ($request->description) {
            $query->where('description', 'like', '%' . $request->description . '%');
        }

        // Obter os pontos com as relações necessárias e ordenar
        $points = $query->with('architect', 'admin')->orderBy('created_at', 'desc')->paginate(10);

        // Obter a lista de arquitetos e administradores para os filtros
        $architects = User::where('role', 'architect')->get();
        $admins = User::where('role', 'admin')->get();

        return view('manager.points.index', compact('points', 'architects', 'admins'));
    }


    public function create()
    {
        $architects = User::where('role', 'architect')->get();
        $scoringRules = ScoringRule::where('is_active', true)->get();

        return view('manager.points.create', compact('architects', 'scoringRules'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'architect_id' => 'required|exists:users,id',
            'scoring_rule_id' => 'required|exists:scoring_rules,id',
            'sale_value' => 'required|numeric|min:0',
            'customer_name' => 'required|string|max:255',
            'description' => 'required|string',
        ]);

        // Obtenha a loja do gerente logado
        $storeId = auth()->user()->storeManagerProfile->store_id;

        // Busca a regra de pontuação selecionada
        $scoringRule = ScoringRule::find($request->scoring_rule_id);

        // Calcula os pontos com base na regra e no valor da venda
        $points = intval($request->sale_value / $scoringRule->amount_spent_per_point);

        // Salva a atribuição de pontos no banco de dados
        Point::create([
            'architect_id' => $request->architect_id,
            'admin_id' => auth()->id(),
            'scoring_rule_id' => $request->scoring_rule_id,
            'scoring_rule_name' => $scoringRule->name,
            'store_id' => $storeId,  // Atribui a loja do gerente logado
            'points' => $points,
            'sale_value' => $request->sale_value,
            'customer_name' => $request->customer_name,
            'description' => $request->description,
        ]);

        return redirect()->route('manager.points.index')->with('success', 'Pontos atribuídos com sucesso!');
    }


    public function edit($id)
    {
        $point = Point::findOrFail($id);
        $architects = User::where('role', 'architect')->get();
        $scoringRules = ScoringRule::where('is_active', true)->get();

        return view('manager.points.edit', compact('point', 'architects', 'scoringRules'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'architect_id' => 'required|exists:users,id',
            'scoring_rule_id' => 'required|exists:scoring_rules,id',
            'sale_value' => 'required|numeric|min:0',
            'customer_name' => 'required|string|max:255',
            'description' => 'required|string',
        ]);

        $point = Point::findOrFail($id);

        // Obtenha a loja do gerente logado
        $storeId = auth()->user()->storeManagerProfile->store_id;

        // Busca a regra de pontuação selecionada
        $scoringRule = ScoringRule::find($request->scoring_rule_id);

        // Calcula os pontos com base na regra e no valor da venda
        $points = intval($request->sale_value / $scoringRule->amount_spent_per_point);

        $point->update([
            'architect_id' => $request->architect_id,
            'scoring_rule_id' => $request->scoring_rule_id,
            'scoring_rule_name' => $scoringRule->name,
            'store_id' => $storeId,  // Atribui a loja do gerente logado
            'points' => $points,
            'sale_value' => $request->sale_value,
            'customer_name' => $request->customer_name,
            'description' => $request->description,
        ]);

        return redirect()->route('manager.points.index')->with('success', 'Ponto atualizado com sucesso!');
    }


    public function destroy($id)
    {
        $point = Point::findOrFail($id);
        $point->delete();
        return redirect()->route('manager.points.index')->with('success', 'Ponto excluído com sucesso!');
    }



    // Parte de pontos para arquitetos
    public function assignPoints(User $architect)
    {
        $scoringRules = ScoringRule::where('is_active', true)->get();

        return view('manager.points.assign', compact('architect', 'scoringRules'));
    }

    public function storePoints(Request $request, User $architect)
    {
        $request->validate([
            'scoring_rule_id' => 'required|exists:scoring_rules,id',
            'sale_value' => 'required|numeric',
            'customer_name' => 'required|string|max:255',
            'description' => 'required|string',
        ]);

        $scoringRule = ScoringRule::find($request->scoring_rule_id);
        $points = intval($request->sale_value / $scoringRule->amount_spent_per_point);

        // Obter a loja do usuário logado
        $storeId = auth()->user()->storeManagerProfile->store_id;

        Point::create([
            'architect_id' => $architect->id,
            'admin_id' => auth()->id(),
            'scoring_rule_id' => $request->scoring_rule_id,
            'scoring_rule_name' => $scoringRule->name,
            'store_id' => $storeId,  // Atribui automaticamente a loja do usuário
            'points' => $points,
            'sale_value' => $request->sale_value,
            'customer_name' => $request->customer_name,
            'description' => $request->description,
        ]);

        return redirect()->route('manager.architects.show', $architect->id)->with('success', 'Pontos atribuídos com sucesso!');
    }


}

