<?php

namespace App\Http\Controllers\Manager;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Point; 
use App\Models\Ranking; 
use App\Models\User; 
use App\Models\Store; 
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    /**
     * Show the manager dashboard.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        Carbon::setLocale('pt_BR'); // Define o local para português do Brasil

        // Obter o mês atual
        $currentMonth = Carbon::now()->month;

        // Obter o ano atual
        $currentYear = Carbon::now()->year;

        // Obter o ID da loja gerenciada pelo usuário logado
        $storeId = auth()->user()->storeManagerProfile->store_id;

        // Obter o nome da loja
        $storeName = Store::where('id', $storeId)->value('name');

        // Obter o total em vendas no mês para a loja
        $totalSales = Point::where('store_id', $storeId)
            ->whereMonth('created_at', $currentMonth)
            ->sum('sale_value');

        // Obter o número de vendas no mês para a loja
        $numberOfSales = Point::where('store_id', $storeId)
            ->whereMonth('created_at', $currentMonth)
            ->count();

        // Obter o total de pontos distribuídos no mês para a loja
        $totalPointsDistributed = Point::where('store_id', $storeId)
            ->whereMonth('created_at', $currentMonth)
            ->sum('points');

        // Obter rankings ativos
        $activeRankings = Ranking::where('is_active', true)->get();

        // Obter os aniversariantes do mês (arquitetos associados à loja)
        $birthdayArchitects = User::where('role', 'architect')
            ->join('architect_profiles', 'users.id', '=', 'architect_profiles.user_id')
            ->whereMonth('architect_profiles.birth_date', $currentMonth)
            ->select('users.id', 'users.name', 'architect_profiles.birth_date')
            ->distinct()
            ->orderBy('architect_profiles.birth_date', 'asc')
            ->get();

        // Obter os últimos 9 meses de dados de vendas para a loja
        $salesData = Point::where('store_id', $storeId)
            ->select(
                DB::raw('SUM(sale_value) as total_sales'),
                DB::raw('COUNT(id) as total_sales_count'),
                DB::raw('MONTH(created_at) as month')
            )
            ->where('created_at', '>=', Carbon::now()->subMonths(9))
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        // Preencher os dados para o gráfico
        $salesByMonth = [];
        $salesCountByMonth = [];
        $months = [];

        for ($i = 8; $i >= 0; $i--) {
            $date = Carbon::now()->subMonths($i);
            $month = $date->translatedFormat('M');
            $months[] = $month;

            $salesForMonth = $salesData->firstWhere('month', $date->month);
            if ($salesForMonth) {
                $salesByMonth[] = $salesForMonth->total_sales;
                $salesCountByMonth[] = $salesForMonth->total_sales_count;
            } else {
                $salesByMonth[] = 0;
                $salesCountByMonth[] = 0;
            }
        }

        // Obter o total de vendas e pontos distribuídos no mês atual para a loja
        $totalSalesThisMonth = Point::where('store_id', $storeId)
            ->whereMonth('created_at', $currentMonth)
            ->sum('sale_value');

        $numberOfSalesThisMonth = Point::where('store_id', $storeId)
            ->whereMonth('created_at', $currentMonth)
            ->count();

        // Obter o total de vendas e pontos distribuídos no ano atual para a loja
        $totalSalesThisYear = Point::where('store_id', $storeId)
            ->whereYear('created_at', $currentYear)
            ->sum('sale_value');

        $numberOfSalesThisYear = Point::where('store_id', $storeId)
            ->whereYear('created_at', $currentYear)
            ->count();

        // Passar os dados para a view
        return view('manager.dashboard', compact(
            'totalSales',
            'numberOfSales',
            'totalPointsDistributed',
            'activeRankings',
            'birthdayArchitects',
            'salesByMonth',
            'salesCountByMonth',
            'months',
            'totalSalesThisMonth',
            'numberOfSalesThisMonth',
            'totalSalesThisYear',
            'numberOfSalesThisYear',
            'storeName'
        ));
    }
}