<?php

namespace App\Http\Controllers\Architect;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Ranking;
use App\Models\User;
use Carbon\Carbon;

class RankingsController extends Controller
{
    public function index()
    {
        // Obter a data atual
        $today = Carbon::now();

        // Buscar rankings ativos e dentro do período de validade
        $activeRankings = Ranking::where('is_active', true)
            ->whereDate('start_date', '<=', $today)
            ->whereDate('end_date', '>=', $today)
            ->orderBy('start_date', 'desc')
            ->get();

        // Retornar a view com os rankings ativos
        return view('architect.rankings.index', compact('activeRankings'));
    }

    public function show($id)
    {
        // Carregar o ranking com os prêmios e a imagem
        $ranking = Ranking::with(['rewards', 'image'])->findOrFail($id);

        // Verificar se o criador do ranking é realmente um admin
        $admin = User::where('id', $ranking->admin_id)->where('role', 'admin')->first();

        /** @var \App\Models\User $user */
        $user = auth()->user();
        $qualifiedRewards = [];
        $userPosition = null;
        $userPoints = 0;

        if ($ranking->type === 'goal') {
            // Verificar se o arquiteto atingiu a meta de pontos para cada prêmio
            $userPoints = $user->points()
                ->whereBetween('created_at', [$ranking->start_date, $ranking->end_date])
                ->sum('points');

            foreach ($ranking->rewards as $reward) {
                if ($userPoints >= $reward->points_required) {
                    $qualifiedRewards[] = $reward;
                }
            }
        } elseif ($ranking->type === 'position') {
            // Calcular a posição do arquiteto no ranking
            $architectsRanking = User::where('role', 'architect')
                ->withSum(['points' => function ($query) use ($ranking) {
                    $query->whereBetween('created_at', [$ranking->start_date, $ranking->end_date]);
                }], 'points')
                ->orderBy('points_sum_points', 'desc')
                ->get();

            $userPosition = $architectsRanking->search(function ($architect) use ($user) {
                return $architect->id === $user->id;
            }) + 1;

            // Também podemos calcular os pontos do usuário no ranking
            $userPoints = $user->points()
                ->whereBetween('created_at', [$ranking->start_date, $ranking->end_date])
                ->sum('points');
        }

        return view('architect.rankings.show', compact('ranking', 'admin', 'qualifiedRewards', 'userPosition', 'userPoints'));
    }
}