<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\ArchitectProfile;
use App\Models\Store;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Contar o total de usuários, admins e gerentes de loja
        $totalUsers = User::whereIn('role', ['admin', 'store_manager'])->count();
        $totalAdmins = User::where('role', 'admin')->count();
        $totalStoreManagers = User::where('role', 'store_manager')->count();

        // Inicializar a query com todos os admins e gerentes de loja
        $query = User::whereIn('role', ['admin', 'store_manager']);

        // Adicionar filtragem por tipo de usuário
        if ($request->has('role') && in_array($request->input('role'), ['admin', 'store_manager'])) {
            $query->where('role', $request->input('role'));
        }

        // Adicionar filtragem e busca por nome ou email
        if ($request->has('search') && $request->input('search') !== null) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->input('search') . '%')
                    ->orWhere('email', 'like', '%' . $request->input('search') . '%');
            });
        }

        $users = $query->paginate(10);

        return view('admin.users.index', compact('users', 'totalUsers', 'totalAdmins', 'totalStoreManagers'));
    }




    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $stores = Store::all();
        return view('admin.users.create', compact('stores'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $rules = [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'role' => 'required|in:admin,architect,store_manager',
        ];

        // Validações adicionais para arquitetos
        if ($request->input('role') == 'architect') {
            $rules = array_merge($rules, [
                'cpf' => 'required|string|unique:architect_profiles',
                'cep' => 'nullable|string',
                'rua' => 'nullable|string',
                'numero' => 'nullable|string',
                'bairro' => 'nullable|string',
                'cidade' => 'nullable|string',
                'estado' => 'nullable|string',
                'phone' => 'nullable|string',
                'birth_date' => 'nullable|date',
            ]);
        }

        // Validações adicionais para gerentes de loja
        if ($request->input('role') == 'store_manager') {
            $rules = array_merge($rules, [
                'store_id' => 'required|exists:stores,id',
                'phone' => 'required|string',
            ]);
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $validatedData = $validator->validated();

        try {
            // Criação do usuário
            $user = User::create([
                'name' => $validatedData['name'],
                'email' => $validatedData['email'],
                'password' => Hash::make($validatedData['password']),
                'role' => $validatedData['role'],
            ]);

            // Criação do perfil de arquiteto
            if ($user->role == 'architect') {
                $architectData = [
                    'cpf' => $validatedData['cpf'],
                    'cep' => $validatedData['cep'],
                    'rua' => $validatedData['rua'],
                    'numero' => $validatedData['numero'],
                    'bairro' => $validatedData['bairro'],
                    'cidade' => $validatedData['cidade'],
                    'estado' => $validatedData['estado'],
                    'phone' => $validatedData['phone'],
                    'birth_date' => $validatedData['birth_date'],
                ];
                $user->architectProfile()->create($architectData);
            }

            // Criação do perfil de gerente de loja
            if ($user->role == 'store_manager') {
                $storeManagerData = [
                    'store_id' => $validatedData['store_id'],
                    'phone' => $validatedData['phone'],
                ];
                $user->storeManagerProfile()->create($storeManagerData);
            }

            return response()->json(['message' => 'Usuário criado com sucesso.'], 201);
        } catch (\Exception $e) {
            Log::error('Erro ao criar usuário: ' . $e->getMessage());
            return response()->json(['message' => 'Erro interno ao criar o usuário.'], 500);
        }
    }


    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        return view('admin.users.show', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    /**
 * Show the form for editing the specified resource.
 */
    public function edit($id)
    {
        $user = User::findOrFail($id);
        $stores = Store::all(); // Para preencher o dropdown de lojas
        return view('admin.users.edit', compact('user', 'stores'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $rules = [
            'name' => 'sometimes|required|string|max:255',
            'email' => 'sometimes|required|string|email|max:255|unique:users,email,' . $user->id,
            'password' => 'nullable|string|min:8|confirmed',
            'role' => 'sometimes|required|in:admin,architect,store_manager',
        ];

        if ($request->input('role') == 'architect') {
            $rules = array_merge($rules, [
                'cpf' => 'sometimes|required|string|unique:architect_profiles,cpf,' . ($user->architectProfile->id ?? 'NULL'),
                'cep' => 'nullable|string',
                'rua' => 'nullable|string',
                'numero' => 'nullable|string',
                'bairro' => 'nullable|string',
                'cidade' => 'nullable|string',
                'estado' => 'nullable|string',
                'phone' => 'nullable|string',
                'birth_date' => 'nullable|date',
            ]);
        }

        if ($request->input('role') == 'store_manager') {
            $rules = array_merge($rules, [
                'store_id' => 'required|exists:stores,id',
                'phone' => 'sometimes|required|string',
            ]);
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $validatedData = $validator->validated();

        try {
            if ($request->filled('password')) {
                $validatedData['password'] = Hash::make($validatedData['password']);
            } else {
                unset($validatedData['password']);
            }

            $user->update($validatedData);

            if ($user->role == 'architect') {
                $user->architectProfile()->updateOrCreate(['user_id' => $user->id], $validatedData);
            }

            if ($user->role == 'store_manager') {
                $storeManagerData = [
                    'store_id' => $validatedData['store_id'],
                    'phone' => $validatedData['phone'],
                ];
                $user->storeManagerProfile()->updateOrCreate(['user_id' => $user->id], $storeManagerData);
            }

            return response()->json(['message' => 'Usuário atualizado com sucesso.'], 200);
        } catch (\Exception $e) {
            Log::error('Erro ao atualizar usuário: ' . $e->getMessage());
            return response()->json(['message' => 'Erro interno ao atualizar o usuário.'], 500);
        }
    }
    
    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        $user->delete();
        return redirect()->route('admin.users.index')->with('success', 'User deleted successfully.');
    }
}
