<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Ranking;
use App\Models\Image;
use App\Models\RankingReward;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class RankingController extends Controller
{
    public function index(Request $request)
    {
        $query = Ranking::query();

        if ($request->has('start_date') && $request->start_date) {
            $query->where('start_date', '>=', $request->start_date);
        }

        if ($request->has('end_date') && $request->end_date) {
            $query->where('end_date', '<=', $request->end_date);
        }

        if ($request->has('is_active') && $request->is_active !== null) {
            $query->where('is_active', $request->is_active);
        }

        $rankings = $query->paginate(10);

        return view('admin.rankings.index', compact('rankings'));
    }

    public function create()
    {
        return view('admin.rankings.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'is_active' => 'required|boolean',
            'featured_image_id' => 'nullable|integer|exists:images,id',
            'type' => 'required|string|in:position,goal',
            'rewards' => 'nullable|array',
            'rewards.*.name' => 'required_with:rewards|string|max:255',
            'rewards.*.points_required' => 'nullable|integer|min:1|required_if:type,goal',
            'rewards.*.position_required' => 'nullable|integer|min:1|required_if:type,position',
            'rewards.*.image_id' => 'nullable|integer|exists:images,id',
        ]);

        $ranking = new Ranking($request->all());
        $ranking->admin_id = Auth::id();
        $ranking->save();

        if ($request->has('rewards')) {
            foreach ($request->rewards as $rewardData) {
                $reward = new RankingReward($rewardData);
                $reward->ranking_id = $ranking->id;
                $reward->save();
            }
        }

        return redirect()->route('admin.rankings.index')->with('success', 'Ranking criado com sucesso.');
    }

    public function edit(Ranking $ranking)
    {
        return view('admin.rankings.edit', compact('ranking'));
    }

    public function update(Request $request, Ranking $ranking)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'is_active' => 'required|boolean',
            'featured_image_id' => 'nullable|integer|exists:images,id',
            'type' => 'required|string|in:position,goal',
            'rewards' => 'nullable|array',
            'rewards.*.name' => 'required_with:rewards|string|max:255',
            'rewards.*.points_required' => 'nullable|integer|min:1|required_if:type,goal',
            'rewards.*.position_required' => 'nullable|integer|min:1|required_if:type,position',
            'rewards.*.image_id' => 'nullable|integer|exists:images,id',
        ]);

        $ranking->update($request->all());

        // Atualizar os prêmios (rewards)
        $ranking->rewards()->delete(); // Remove os prêmios antigos
        if ($request->has('rewards')) {
            foreach ($request->rewards as $rewardData) {
                $reward = new RankingReward($rewardData);
                $reward->ranking_id = $ranking->id;
                $reward->save();
            }
        }

        return redirect()->route('admin.rankings.index')->with('success', 'Ranking atualizado com sucesso.');
    }
    

    public function show(Ranking $ranking)
    {
        // Carregar o ranking junto com os prêmios, imagem destacada e o administrador
        $ranking->load(['rewards' => function ($query) use ($ranking) {
            if ($ranking->type === 'goal') {
                $query->orderByDesc('points_required');
            } elseif ($ranking->type === 'position') {
                $query->orderBy('position_required');
            }
        }, 'image']);

        // Verificar se o criador do ranking é realmente um admin
        $admin = User::where('id', $ranking->admin_id)->where('role', 'admin')->first();

        $qualifiedArchitects = [];
        $architectRanking = [];

        if ($ranking->type === 'goal') {
            foreach ($ranking->rewards as $reward) {
                // Obter arquitetos qualificados para o prêmio
                $qualifiedArchitects[$reward->id] = User::where('role', 'architect')
                    ->withSum(['points' => function ($query) use ($ranking) {
                        $query->whereBetween('created_at', [$ranking->start_date, $ranking->end_date]);
                    }], 'points')
                    ->having('points_sum_points', '>=', $reward->points_required)
                    ->get();
            }
        } elseif ($ranking->type === 'position') {
            // Obter ranking de arquitetos baseado na soma dos pontos
            $architectRanking = User::where('role', 'architect')
                ->withSum(['points' => function ($query) use ($ranking) {
                    $query->whereBetween('created_at', [$ranking->start_date, $ranking->end_date]);
                }], 'points')
                ->orderBy('points_sum_points', 'desc') // Ordena por soma de pontos em ordem decrescente
                ->get();

            // Associar cada prêmio ao arquiteto que ocupa a posição necessária
            foreach ($ranking->rewards as $reward) {
                $reward->architect = $architectRanking->get($reward->position_required - 1); // Posição necessária menos 1 porque a lista é baseada em índice zero
            }
        }

        return view('admin.rankings.show', compact('ranking', 'admin', 'qualifiedArchitects', 'architectRanking'));
    }

    public function destroy(Ranking $ranking)
    {
        $ranking->delete();
        return redirect()->route('admin.rankings.index')->with('success', 'Ranking excluído com sucesso.');
    }
}
