<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Point;
use App\Models\ScoringRule;
use App\Models\Store;
use App\Models\User;

class PointsController extends Controller
{
    public function index(Request $request)
    {
        $query = Point::query();
    
        // Filtros
        if ($request->architect_id) {
            $query->where('architect_id', $request->architect_id);
        }
    
        if ($request->store_id) {
            $query->where('store_id', $request->store_id);
        }
    
        if ($request->start_date) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }
    
        if ($request->end_date) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }
    
        // Calcula os totais sem a paginação, para pegar todos os registros que atendem aos filtros
        $totalPoints = $query->sum('points');
        $totalSaleValue = $query->sum('sale_value');
    
        // Pagina os resultados filtrados
        $points = $query->with('architect', 'store', 'admin')->orderBy('created_at', 'desc')->paginate(10);
    
        // Obter arquitetos e lojas para os filtros
        $architects = User::where('role', 'architect')->get();
        $stores = Store::all();
    
        return view('admin.points.index', compact('points', 'architects', 'stores', 'totalPoints', 'totalSaleValue'));
    }


    public function create()
    {
        $architects = User::where('role', 'architect')->get();
        $scoringRules = ScoringRule::where('is_active', true)->get();
        $stores = Store::all();

        return view('admin.points.create', compact('architects', 'scoringRules', 'stores'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'architect_id' => 'required|exists:users,id',
            'scoring_rule_id' => 'required|exists:scoring_rules,id',
            'store_id' => 'required|exists:stores,id',
            'sale_value' => 'required|numeric|min:0',
            'customer_name' => 'required|string|max:255',
            'description' => 'required|string',
        ]);

        // Busca a regra de pontuação selecionada
        $scoringRule = ScoringRule::find($request->scoring_rule_id);

        // Calcula os pontos com base na regra e no valor da venda
        $points = intval($request->sale_value / $scoringRule->amount_spent_per_point);

        // Salva a atribuição de pontos no banco de dados
        Point::create([
            'architect_id' => $request->architect_id,
            'admin_id' => auth()->id(),
            'scoring_rule_id' => $request->scoring_rule_id,
            'scoring_rule_name' => $scoringRule->name,
            'store_id' => $request->store_id,
            'points' => $points,
            'sale_value' => $request->sale_value,
            'customer_name' => $request->customer_name,
            'description' => $request->description,
        ]);

        return redirect()->route('admin.points.index')->with('success', 'Pontos atribuídos com sucesso!');
    }

    public function edit($id)
    {
        $point = Point::findOrFail($id);
        $architects = User::where('role', 'architect')->get();
        $scoringRules = ScoringRule::where('is_active', true)->get();
        $stores = Store::all();

        return view('admin.points.edit', compact('point', 'architects', 'scoringRules', 'stores'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'architect_id' => 'required|exists:users,id',
            'scoring_rule_id' => 'required|exists:scoring_rules,id',
            'store_id' => 'required|exists:stores,id',
            'sale_value' => 'required|numeric',
            'customer_name' => 'required|string|max:255',
            'description' => 'required|string',
        ]);

        $point = Point::findOrFail($id);

        // Busca a regra de pontuação selecionada
        $scoringRule = ScoringRule::find($request->scoring_rule_id);

        // Calcula os pontos com base na regra e no valor da venda
        $points = intval($request->sale_value / $scoringRule->amount_spent_per_point);

        $point->update([
            'architect_id' => $request->architect_id,
            'scoring_rule_id' => $request->scoring_rule_id,
            'scoring_rule_name' => $scoringRule->name,
            'store_id' => $request->store_id,
            'points' => $points,
            'sale_value' => $request->sale_value,
            'customer_name' => $request->customer_name,
            'description' => $request->description,
        ]);

        return redirect()->route('admin.points.index')->with('success', 'Ponto atualizado com sucesso!');
    }

    public function destroy($id)
    {
        $point = Point::findOrFail($id);
        $point->delete();
        return redirect()->route('admin.points.index')->with('success', 'Ponto excluído com sucesso!');
    }



    // Parte de pontos para arquitetos
    public function assignPoints(User $architect)
    {
        $scoringRules = ScoringRule::where('is_active', true)->get();
        $stores = Store::all();

        return view('admin.points.assign', compact('architect', 'scoringRules', 'stores'));
    }

    public function storePoints(Request $request, User $architect)
    {
        $request->validate([
            'scoring_rule_id' => 'required|exists:scoring_rules,id',
            'store_id' => 'required|exists:stores,id',
            'sale_value' => 'required|numeric',
            'customer_name' => 'required|string|max:255',
            'description' => 'required|string',
        ]);

        $scoringRule = ScoringRule::find($request->scoring_rule_id);
        $points = intval($request->sale_value / $scoringRule->amount_spent_per_point);

        Point::create([
            'architect_id' => $architect->id,
            'admin_id' => auth()->id(),
            'scoring_rule_id' => $request->scoring_rule_id,
            'scoring_rule_name' => $scoringRule->name,
            'store_id' => $request->store_id,
            'points' => $points,
            'sale_value' => $request->sale_value,
            'customer_name' => $request->customer_name,
            'description' => $request->description,
        ]);

        return redirect()->route('admin.architects.show', $architect->id)->with('success', 'Pontos atribuídos com sucesso!');
    }

}

