<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\ArchitectProfile;
use Carbon\Carbon;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;


class ArchitectController extends Controller
{
    public function index(Request $request)
    {
        $totalArchitects = User::where('role', 'architect')->count();

        $query = User::query()->where('role', 'architect');

        // Adicionar filtragem e busca
        if ($request->has('search') && $request->input('search') !== null) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->input('search') . '%')
                    ->orWhere('email', 'like', '%' . $request->input('search') . '%');
            });
        }

        // Ordenar por nome em ordem alfabética
        $query->orderBy('name');

        $architects = $query->with('architectProfile')->paginate(10);

        return view('admin.architects.index', compact('architects', 'totalArchitects'));
    }


    public function show(User $architect)
    {
        $currentMonth = Carbon::now()->month;
        $currentYear = Carbon::now()->year;

        // Calcular pontos ganhos no mês
        $pointsThisMonth = $architect->points()
            ->whereMonth('created_at', $currentMonth)
            ->whereYear('created_at', $currentYear)
            ->sum('points');

        // Calcular pontos ganhos no ano
        $pointsThisYear = $architect->points()
            ->whereYear('created_at', $currentYear)
            ->sum('points');

        // Calcular valor de vendas no mês
        $salesThisMonth = $architect->points()
            ->whereMonth('created_at', $currentMonth)
            ->whereYear('created_at', $currentYear)
            ->sum('sale_value');

        // Calcular valor de vendas no ano
        $salesThisYear = $architect->points()
            ->whereYear('created_at', $currentYear)
            ->sum('sale_value');

        return view('admin.architects.show', compact('architect', 'pointsThisMonth', 'pointsThisYear', 'salesThisMonth', 'salesThisYear'));
    }

    public function create()
    {
        return view('admin.architects.create');
    }

    public function store(Request $request)
    {
        $rules = [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'cpf' => 'required|string|unique:architect_profiles',
            'cep' => 'nullable|string',
            'rua' => 'nullable|string',
            'numero' => 'nullable|string',
            'bairro' => 'nullable|string',
            'cidade' => 'nullable|string',
            'estado' => 'nullable|string',
            'phone' => 'nullable|string',
            'birth_date' => 'nullable|date',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $validatedData = $validator->validated();

        try {
            $user = User::create([
                'name' => $validatedData['name'],
                'email' => $validatedData['email'],
                'password' => Hash::make($validatedData['password']),
                'role' => 'architect',
            ]);

            $architectData = [
                'cpf' => $validatedData['cpf'],
                'cep' => $validatedData['cep'],
                'rua' => $validatedData['rua'],
                'numero' => $validatedData['numero'],
                'bairro' => $validatedData['bairro'],
                'cidade' => $validatedData['cidade'],
                'estado' => $validatedData['estado'],
                'phone' => $validatedData['phone'],
                'birth_date' => $validatedData['birth_date'],
            ];
            $user->architectProfile()->create($architectData);

            return response()->json(['message' => 'Arquiteto criado com sucesso.', 'id' => $user->id], 201);
        } catch (\Exception $e) {
            Log::error('Erro ao criar arquiteto: ' . $e->getMessage());
            return response()->json(['message' => 'Erro interno ao criar o arquiteto.'], 500);
        }
    }

    public function destroy(User $architect)
    {
        $architect->delete();
        return redirect()->route('admin.architects.index')->with('success', 'Arquiteto deletado com sucesso.');
    }

    public function edit(User $architect)
    {
        return view('admin.architects.edit', compact('architect'));
    }

    public function update(Request $request, User $architect)
    {
        $rules = [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $architect->id,
            'password' => 'nullable|string|min:8|confirmed',
            'cpf' => 'required|string|unique:architect_profiles,cpf,' . $architect->architectProfile->id,
            'cep' => 'nullable|string',
            'rua' => 'nullable|string',
            'numero' => 'nullable|string',
            'bairro' => 'nullable|string',
            'cidade' => 'nullable|string',
            'estado' => 'nullable|string',
            'phone' => 'nullable|string',
            'birth_date' => 'nullable|date',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $validatedData = $validator->validated();

        try {
            $architect->update([
                'name' => $validatedData['name'],
                'email' => $validatedData['email'],
                'password' => $validatedData['password'] ? Hash::make($validatedData['password']) : $architect->password,
            ]);

            $architect->architectProfile()->update([
                'cpf' => $validatedData['cpf'],
                'cep' => $validatedData['cep'],
                'rua' => $validatedData['rua'],
                'numero' => $validatedData['numero'],
                'bairro' => $validatedData['bairro'],
                'cidade' => $validatedData['cidade'],
                'estado' => $validatedData['estado'],
                'phone' => $validatedData['phone'],
                'birth_date' => $validatedData['birth_date'],
            ]);

            return response()->json(['message' => 'Arquiteto atualizado com sucesso.', 'id' => $architect->id], 200);
        } catch (\Exception $e) {
            Log::error('Erro ao atualizar arquiteto: ' . $e->getMessage());
            return response()->json(['message' => 'Erro interno ao atualizar o arquiteto.'], 500);
        }
    }

}